const express = require('express')
const { withGoogle } = require('../google')
const { auth } = require('../auth')

const router = express.Router()

router.get('/labels', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      const r = await gmail.users.labels.list({ userId: 'me' })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.get('/messages', auth, async (req, res) => {
  try {
    const q = req.query.q || ''
    const labelIds = req.query.labelIds ? [].concat(req.query.labelIds) : undefined
    const pageToken = req.query.pageToken || undefined
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      const r = await gmail.users.messages.list({ userId: 'me', q, labelIds, pageToken, maxResults: 25 })
      const msgs = r.data.messages || []
      const details = await Promise.all(
        msgs.map(m => gmail.users.messages.get({ userId: 'me', id: m.id, format: 'metadata', metadataHeaders: ['Subject', 'From', 'To', 'Date'] }))
      )
      return { nextPageToken: r.data.nextPageToken || null, messages: details.map(x => x.data) }
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.get('/messages/:id', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      const r = await gmail.users.messages.get({ userId: 'me', id: req.params.id, format: 'full' })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/messages/:id/delete', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      await gmail.users.messages.delete({ userId: 'me', id: req.params.id })
      return { ok: true }
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/messages/:id/archive', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      await gmail.users.messages.modify({ userId: 'me', id: req.params.id, requestBody: { removeLabelIds: ['INBOX'] } })
      return { ok: true }
    })
    res.json(out)
  } catch (e) {
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/send', auth, async (req, res) => {
  try {
    const to = req.body.to || ''
    const subject = req.body.subject || ''
    const body = req.body.body || ''
    const cc = req.body.cc || ''
    const bcc = req.body.bcc || ''
    const contentType = req.body.contentType || 'text/plain; charset=UTF-8'
    const raw = buildRawMail({ to, subject, body, cc, bcc, contentType })
    const out = await withGoogle(req.user.uid, async ({ gmail }) => {
      const r = await gmail.users.messages.send({ userId: 'me', requestBody: { raw } })
      return r.data
    })
    res.json(out)
  } catch (e) {
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

function buildRawMail({ to, subject, body, cc, bcc, contentType }) {
  const lines = []
  if (to) lines.push('To: ' + to)
  if (cc) lines.push('Cc: ' + cc)
  if (bcc) lines.push('Bcc: ' + bcc)
  lines.push('Subject: ' + subject)
  lines.push('Content-Type: ' + contentType)
  lines.push('')
  lines.push(body)
  const str = lines.join('\r\n')
  return Buffer.from(str).toString('base64').replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '')
}

function statusFromError(e) {
  const code = e.code || e.status || 500
  if (code === 429) return 429
  return typeof code === 'number' ? code : 500
}

function errorBody(e) {
  return { error: (e && e.message) || 'error' }
}

function maybeRevoke(req, e) {
  const code = e && (e.code || e.status)
  const msg = (e && e.message) || ''
  if (code === 401 && msg.includes('invalid_grant')) {
    console.warn('[Gmail] Revoking tokens due to invalid_grant for user:', req.user.uid)
    try { require('../db').revokeOauthToken(req.user.uid, 'google') } catch (_) {}
  }
}

module.exports = router
