require('dotenv').config()
const { google } = require('googleapis')
const { encrypt } = require('./crypto')
const { upsertOauthToken, getOauthToken, selectById } = require('./db')

const clientId = process.env.GOOGLE_OAUTH_CLIENT_ID
const clientSecret = process.env.GOOGLE_OAUTH_CLIENT_SECRET
const redirectUri = process.env.GOOGLE_OAUTH_REDIRECT_URI

function getOauth2Client(userId) {
  const oauth2Client = new google.auth.OAuth2(clientId, clientSecret, redirectUri)
  const tokenRow = getOauthToken(userId, 'google')
  if (tokenRow && tokenRow.access_token) {
    const creds = { access_token: tokenRow.access_token, token_type: tokenRow.token_type || 'Bearer' }
    if (tokenRow.refresh_token_enc) creds.refresh_token = require('./crypto').decrypt(tokenRow.refresh_token_enc)
    if (tokenRow.expiry_date) creds.expiry_date = tokenRow.expiry_date
    oauth2Client.setCredentials(creds)
  }
  return oauth2Client
}

function authUrl(scopes, state) {
  const oauth2Client = new google.auth.OAuth2(clientId, clientSecret, redirectUri)
  return oauth2Client.generateAuthUrl({
    access_type: 'offline',
    prompt: 'consent',
    scope: scopes,
    state
  })
}

async function handleCallback(userId, code) {
  const oauth2Client = new google.auth.OAuth2(clientId, clientSecret, redirectUri)
  const { tokens } = await oauth2Client.getToken(code)
  const u = selectById.get(userId)
  if (!u) throw new Error('user_not_found')
  const org = process.env.ALLOWED_ORGANIZATION || ''
  if (org && u.email && !u.email.endsWith('@' + org)) {
    const msg = `Access blocked: user email domain not in allowed organization '${org}'`
    console.error('[OAuth] Organization access denied on callback:', { userId, email: u.email, allowed: org })
    throw new Error(msg)
  }
  const payload = {
    scope: Array.isArray(tokens.scope) ? tokens.scope.join(' ') : tokens.scope || null,
    access_token: tokens.access_token || null,
    refresh_token_enc: tokens.refresh_token ? encrypt(tokens.refresh_token) : null,
    token_type: tokens.token_type || 'Bearer',
    expiry_date: tokens.expiry_date || null,
    revoked_at: null
  }
  const row = upsertOauthToken(u.id, 'google', payload)
  return row
}

module.exports = { getOauth2Client, authUrl, handleCallback }
