const Database = require('better-sqlite3')
const path = require('path')

const dbPath = path.resolve(process.env.DATABASE_URL || './data/app.db')
const db = new Database(dbPath)

db.exec(
  `CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    google_sub TEXT UNIQUE NOT NULL,
    email TEXT UNIQUE,
    name TEXT,
    avatar_url TEXT,
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    last_login_at TEXT
  );`
)

db.exec(
  `CREATE TABLE IF NOT EXISTS oauth_tokens (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    user_id INTEGER NOT NULL,
    provider TEXT NOT NULL,
    scope TEXT,
    access_token TEXT,
    refresh_token_enc TEXT,
    token_type TEXT,
    expiry_date INTEGER,
    revoked_at TEXT,
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    FOREIGN KEY(user_id) REFERENCES users(id)
  );`
)

const insertUser = db.prepare(
  'INSERT INTO users (google_sub, email, name, avatar_url, created_at, updated_at, last_login_at) VALUES (?, ?, ?, ?, ?, ?, ?)'
)
const updateUser = db.prepare(
  'UPDATE users SET email = COALESCE(?, email), name = COALESCE(?, name), avatar_url = COALESCE(?, avatar_url), updated_at = ?, last_login_at = ? WHERE google_sub = ?'
)
const selectBySub = db.prepare('SELECT * FROM users WHERE google_sub = ?')
const selectById = db.prepare('SELECT * FROM users WHERE id = ?')
const existsBySub = db.prepare('SELECT 1 FROM users WHERE google_sub = ? LIMIT 1')

const insertToken = db.prepare(
  'INSERT INTO oauth_tokens (user_id, provider, scope, access_token, refresh_token_enc, token_type, expiry_date, revoked_at, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
)
const updateToken = db.prepare(
  'UPDATE oauth_tokens SET scope = COALESCE(?, scope), access_token = ?, refresh_token_enc = COALESCE(?, refresh_token_enc), token_type = COALESCE(?, token_type), expiry_date = COALESCE(?, expiry_date), revoked_at = ?, updated_at = ? WHERE user_id = ? AND provider = ?'
)
const selectTokenByUser = db.prepare('SELECT * FROM oauth_tokens WHERE user_id = ? AND provider = ?')

function nowIso() {
  return new Date().toISOString()
}

function upsertUserFromGoogle(payload) {
  const sub = payload.sub
  const email = payload.email || null
  const name = payload.name || null
  const avatar = payload.picture || null
  const existing = selectBySub.get(sub)
  const ts = nowIso()
  if (!existing) {
    insertUser.run(sub, email, name, avatar, ts, ts, ts)
    return selectBySub.get(sub)
  } else {
    updateUser.run(email, name, avatar, ts, ts, sub)
    return selectBySub.get(sub)
  }
}

function userExistsByUid(uid) {
  const row = existsBySub.get(uid)
  return !!row
}

function upsertOauthToken(userId, provider, payload) {
  const current = selectTokenByUser.get(userId, provider)
  const ts = nowIso()
  const scope = payload.scope || null
  const access_token = payload.access_token || null
  const refresh_token_enc = payload.refresh_token_enc || null
  const token_type = payload.token_type || null
  const expiry_date = payload.expiry_date || null
  const revoked_at = payload.revoked_at || null
  if (!current) {
    insertToken.run(userId, provider, scope, access_token, refresh_token_enc, token_type, expiry_date, revoked_at, ts, ts)
    return selectTokenByUser.get(userId, provider)
  } else {
    updateToken.run(scope, access_token, refresh_token_enc, token_type, expiry_date, revoked_at, ts, userId, provider)
    return selectTokenByUser.get(userId, provider)
  }
}

function getOauthToken(userId, provider) {
  return selectTokenByUser.get(userId, provider)
}

function revokeOauthToken(userId, provider) {
  const ts = nowIso()
  updateToken.run(null, null, null, null, null, ts, ts, userId, provider)
  return selectTokenByUser.get(userId, provider)
}

module.exports = { db, upsertUserFromGoogle, userExistsByUid, selectById, selectBySub, upsertOauthToken, getOauthToken, revokeOauthToken }
