require('dotenv').config()
const { OAuth2Client } = require('google-auth-library')
const jwt = require('jsonwebtoken')

const googleClient = new OAuth2Client(process.env.GOOGLE_OAUTH_CLIENT_ID)

async function verifyGoogleIdToken(idToken) {
  try {
    const ticket = await googleClient.verifyIdToken({ idToken, audience: process.env.GOOGLE_OAUTH_CLIENT_ID })
    const p = ticket.getPayload()
    const org = process.env.ALLOWED_ORGANIZATION || ''
    if (org && p.hd !== org) {
      const msg = `Access blocked: user domain '${p.hd}' not in allowed organization '${org}'`
      console.error('[Auth] Organization access denied:', { email: p.email, hd: p.hd, allowed: org })
      throw new Error(msg)
    }
    return { sub: p.sub, email: p.email, name: p.name, picture: p.picture }
  } catch (e) {
    if (e.message.includes('org_internal')) {
      console.error('[Auth] Google org_internal error - OAuth app may be configured as Internal in Google Cloud Console:', e.message)
    }
    throw e
  }
}

function issueJwt(user) {
  return jwt.sign({ uid: user.id, email: user.email }, process.env.APP_JWT_SECRET, { expiresIn: process.env.JWT_EXPIRES_IN || '7d' })
}

function auth(req, res, next) {
  const h = req.headers.authorization || ''
  const t = h.startsWith('Bearer ') ? h.slice(7) : ''
  try {
    req.user = jwt.verify(t, process.env.APP_JWT_SECRET)
    console.debug('[Auth] JWT verified for user:', req.user.uid)
    next()
  } catch (e) {
    console.warn('[Auth] JWT verification failed:', e.message)
    res.status(401).json({ error: 'unauthorized' })
  }
}

module.exports = { verifyGoogleIdToken, issueJwt, auth }
